<?php
/**
 * NASS Admin Class
 * 
 * Handles admin functionality for NASS Payment Gateway
 *
 * @package NassPaymentGateway
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Nass_Admin Class
 */
class Nass_Admin {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'init_admin'));
        add_action('wp_ajax_nass_test_connection', array($this, 'test_connection'));
        add_action('wp_ajax_nass_clear_logs', array($this, 'clear_logs'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __('NASS Payment Gateway', 'nass-payment-gateway'),
            __('NASS Gateway', 'nass-payment-gateway'),
            'manage_woocommerce',
            'nass-payment-gateway',
            array($this, 'admin_page')
        );
    }
    
    /**
     * Initialize admin
     */
    public function init_admin() {
        // Add admin notices
        add_action('admin_notices', array($this, 'admin_notices'));
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'nass-payment-gateway') !== false) {
            wp_enqueue_script('nass-admin-js', NASS_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), NASS_PLUGIN_VERSION, true);
            wp_localize_script('nass-admin-js', 'nass_admin_params', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('nass_admin_nonce')
            ));
        }
    }
    
    /**
     * Admin page callback
     */
    public function admin_page() {
        $active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'dashboard';
        
        ?>
        <div class="wrap">
            <h1><?php _e('NASS Payment Gateway', 'nass-payment-gateway'); ?></h1>
            
            <nav class="nav-tab-wrapper">
                <a href="?page=nass-payment-gateway&tab=dashboard" class="nav-tab <?php echo $active_tab == 'dashboard' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Dashboard', 'nass-payment-gateway'); ?>
                </a>
                <a href="?page=nass-payment-gateway&tab=transactions" class="nav-tab <?php echo $active_tab == 'transactions' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Transactions', 'nass-payment-gateway'); ?>
                </a>
                <a href="?page=nass-payment-gateway&tab=logs" class="nav-tab <?php echo $active_tab == 'logs' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Logs', 'nass-payment-gateway'); ?>
                </a>
                <a href="?page=nass-payment-gateway&tab=settings" class="nav-tab <?php echo $active_tab == 'settings' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Settings', 'nass-payment-gateway'); ?>
                </a>
            </nav>
            
            <div class="tab-content">
                <?php
                switch ($active_tab) {
                    case 'transactions':
                        $this->transactions_tab();
                        break;
                    case 'logs':
                        $this->logs_tab();
                        break;
                    case 'settings':
                        $this->settings_tab();
                        break;
                    default:
                        $this->dashboard_tab();
                        break;
                }
                ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Dashboard tab
     */
    private function dashboard_tab() {
        $gateway = new WC_Nass_Payment_Gateway();
        $logger = new Nass_Logger();
        
        // Get some statistics
        global $wpdb;
        $table_name = $wpdb->prefix . 'nass_payment_logs';
        
        $total_transactions = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
        $successful_transactions = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE status = 'Approved'");
        $failed_transactions = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE status != 'Approved'");
        $total_amount = $wpdb->get_var("SELECT SUM(amount) FROM $table_name WHERE status = 'Approved'");
        
        ?>
        <div class="nass-dashboard">
            <div class="nass-stats-grid">
                <div class="nass-stat-card">
                    <h3><?php _e('Total Transactions', 'nass-payment-gateway'); ?></h3>
                    <div class="stat-number"><?php echo number_format($total_transactions ?: 0); ?></div>
                </div>
                <div class="nass-stat-card success">
                    <h3><?php _e('Successful', 'nass-payment-gateway'); ?></h3>
                    <div class="stat-number"><?php echo number_format($successful_transactions ?: 0); ?></div>
                </div>
                <div class="nass-stat-card failed">
                    <h3><?php _e('Failed', 'nass-payment-gateway'); ?></h3>
                    <div class="stat-number"><?php echo number_format($failed_transactions ?: 0); ?></div>
                </div>
                <div class="nass-stat-card total">
                    <h3><?php _e('Total Amount', 'nass-payment-gateway'); ?></h3>
                    <div class="stat-number"><?php echo wc_price($total_amount ?: 0); ?></div>
                </div>
            </div>
            
            <div class="nass-status-section">
                <h3><?php _e('Gateway Status', 'nass-payment-gateway'); ?></h3>
                <table class="wp-list-table widefat fixed striped">
                    <tbody>
                        <tr>
                            <td><strong><?php _e('Status', 'nass-payment-gateway'); ?></strong></td>
                            <td>
                                <?php if ('yes' === $gateway->enabled): ?>
                                    <span class="status-enabled"><?php _e('Enabled', 'nass-payment-gateway'); ?></span>
                                <?php else: ?>
                                    <span class="status-disabled"><?php _e('Disabled', 'nass-payment-gateway'); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('Environment', 'nass-payment-gateway'); ?></strong></td>
                            <td><?php echo ucfirst($gateway->environment); ?></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('Terminal ID', 'nass-payment-gateway'); ?></strong></td>
                            <td><?php echo esc_html($gateway->terminal_id); ?></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('Title', 'nass-payment-gateway'); ?></strong></td>
                            <td><?php echo esc_html($gateway->title); ?></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('Gateway URL', 'nass-payment-gateway'); ?></strong></td>
                            <td><?php echo esc_html($gateway->gateway_url); ?></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('Debug Mode', 'nass-payment-gateway'); ?></strong></td>
                            <td><?php echo $gateway->debug ? __('Enabled', 'nass-payment-gateway') : __('Disabled', 'nass-payment-gateway'); ?></td>
                        </tr>
                    </tbody>
                </table>
                
                <div class="nass-actions">
                    <button type="button" class="button button-secondary" id="test-connection">
                        <?php _e('Test Connection', 'nass-payment-gateway'); ?>
                    </button>
                    <a href="<?php echo admin_url('admin.php?page=wc-settings&tab=checkout&section=nass_gateway'); ?>" class="button button-primary">
                        <?php _e('Configure Gateway', 'nass-payment-gateway'); ?>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Transactions tab
     */
    private function transactions_tab() {
        $logger = new Nass_Logger();
        $transactions = $logger->get_transaction_logs(null, 50);
        
        ?>
        <div class="nass-transactions">
            <h3><?php _e('Recent Transactions', 'nass-payment-gateway'); ?></h3>
            
            <?php if (empty($transactions)): ?>
                <p><?php _e('No transactions found.', 'nass-payment-gateway'); ?></p>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('Order ID', 'nass-payment-gateway'); ?></th>
                            <th><?php _e('Transaction ID', 'nass-payment-gateway'); ?></th>
                            <th><?php _e('Amount', 'nass-payment-gateway'); ?></th>
                            <th><?php _e('Currency', 'nass-payment-gateway'); ?></th>
                            <th><?php _e('Status', 'nass-payment-gateway'); ?></th>
                            <th><?php _e('Date', 'nass-payment-gateway'); ?></th>
                            <th><?php _e('Actions', 'nass-payment-gateway'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($transactions as $transaction): ?>
                            <tr>
                                <td>
                                    <a href="<?php echo admin_url('post.php?post=' . $transaction->order_id . '&action=edit'); ?>">
                                        #<?php echo esc_html($transaction->order_id); ?>
                                    </a>
                                </td>
                                <td><?php echo esc_html($transaction->transaction_id); ?></td>
                                <td><?php echo wc_price($transaction->amount); ?></td>
                                <td><?php echo esc_html($transaction->currency); ?></td>
                                <td>
                                    <span class="status-<?php echo sanitize_html_class(strtolower($transaction->status)); ?>">
                                        <?php echo esc_html($transaction->status); ?>
                                    </span>
                                </td>
                                <td><?php echo esc_html($transaction->created_at); ?></td>
                                <td>
                                    <button type="button" class="button button-small view-details" 
                                            data-transaction='<?php echo esc_attr($transaction->gateway_response); ?>'>
                                        <?php _e('View Details', 'nass-payment-gateway'); ?>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        
        <!-- Transaction Details Modal -->
        <div id="transaction-details-modal" class="nass-modal" style="display: none;">
            <div class="nass-modal-content">
                <span class="nass-modal-close">&times;</span>
                <h3><?php _e('Transaction Details', 'nass-payment-gateway'); ?></h3>
                <div id="transaction-details-content"></div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Logs tab
     */
    private function logs_tab() {
        $logger = new Nass_Logger();
        $log_contents = $logger->get_log_contents(200);
        
        ?>
        <div class="nass-logs">
            <div class="nass-logs-header">
                <h3><?php _e('Debug Logs', 'nass-payment-gateway'); ?></h3>
                <button type="button" class="button button-secondary" id="clear-logs">
                    <?php _e('Clear Logs', 'nass-payment-gateway'); ?>
                </button>
            </div>
            
            <div class="nass-log-viewer">
                <textarea readonly><?php echo esc_textarea($log_contents); ?></textarea>
            </div>
            
            <p class="description">
                <?php _e('Enable debug mode in gateway settings to see detailed logs here.', 'nass-payment-gateway'); ?>
            </p>
        </div>
        <?php
    }
    
    /**
     * Settings tab
     */
    private function settings_tab() {
        ?>
        <div class="nass-settings">
            <h3><?php _e('Gateway Configuration', 'nass-payment-gateway'); ?></h3>
            <p><?php _e('Gateway settings can be configured in the WooCommerce payment settings.', 'nass-payment-gateway'); ?></p>
            
            <div class="nass-settings-cards">
                <div class="settings-card">
                    <h4><?php _e('Basic Settings', 'nass-payment-gateway'); ?></h4>
                    <ul>
                        <li><?php _e('Enable/Disable Gateway', 'nass-payment-gateway'); ?></li>
                        <li><?php _e('Payment Method Title', 'nass-payment-gateway'); ?></li>
                        <li><?php _e('Customer Description', 'nass-payment-gateway'); ?></li>
                        <li><?php _e('Environment Selection', 'nass-payment-gateway'); ?></li>
                    </ul>
                </div>
                
                <div class="settings-card">
                    <h4><?php _e('NASS Credentials', 'nass-payment-gateway'); ?></h4>
                    <ul>
                        <li><?php _e('Terminal ID', 'nass-payment-gateway'); ?></li>
                        <li><?php _e('Username', 'nass-payment-gateway'); ?></li>
                        <li><?php _e('Password', 'nass-payment-gateway'); ?></li>
                        <li><?php _e('Gateway URL', 'nass-payment-gateway'); ?></li>
                    </ul>
                </div>
                
                <div class="settings-card">
                    <h4><?php _e('Advanced Options', 'nass-payment-gateway'); ?></h4>
                    <ul>
                        <li><?php _e('Debug Logging', 'nass-payment-gateway'); ?></li>
                        <li><?php _e('Transaction Timeout', 'nass-payment-gateway'); ?></li>
                        <li><?php _e('Webhook Configuration', 'nass-payment-gateway'); ?></li>
                        <li><?php _e('Security Settings', 'nass-payment-gateway'); ?></li>
                    </ul>
                </div>
            </div>
            
            <a href="<?php echo admin_url('admin.php?page=wc-settings&tab=checkout&section=nass_gateway'); ?>" class="button button-primary">
                <?php _e('Open Gateway Settings', 'nass-payment-gateway'); ?>
            </a>
        </div>
        <?php
    }
    
    /**
     * Test connection AJAX handler
     */
    public function test_connection() {
        check_ajax_referer('nass_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'nass-payment-gateway'));
        }
        
        try {
            $gateway = new WC_Nass_Payment_Gateway();
            $api_handler = new Nass_API_Handler($gateway);
            
            $result = $api_handler->authenticate();
            
            if ($result) {
                wp_send_json_success(array(
                    'message' => __('Connection successful! Authentication with NASS gateway completed.', 'nass-payment-gateway')
                ));
            } else {
                wp_send_json_error(array(
                    'message' => __('Connection failed. Please check your credentials and try again.', 'nass-payment-gateway')
                ));
            }
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Connection error: %s', 'nass-payment-gateway'), $e->getMessage())
            ));
        }
    }
    
    /**
     * Clear logs AJAX handler
     */
    public function clear_logs() {
        check_ajax_referer('nass_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'nass-payment-gateway'));
        }
        
        $logger = new Nass_Logger();
        $logger->clear_log_file();
        
        // Also clear database logs older than 30 days
        $logger->clear_old_logs(30);
        
        wp_send_json_success(array(
            'message' => __('Logs cleared successfully!', 'nass-payment-gateway')
        ));
    }
    
    /**
     * Admin notices
     */
    public function admin_notices() {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            echo '<div class="notice notice-error"><p><strong>' . __('NASS Payment Gateway', 'nass-payment-gateway') . ':</strong> ' . 
                 __('WooCommerce is required for this plugin to work.', 'nass-payment-gateway') . '</p></div>';
            return;
        }
        
        // Check if gateway is configured
        $gateway = new WC_Nass_Payment_Gateway();
        if ('yes' === $gateway->enabled && (empty($gateway->terminal_id) || empty($gateway->username) || empty($gateway->password))) {
            echo '<div class="notice notice-warning"><p><strong>' . __('NASS Payment Gateway', 'nass-payment-gateway') . ':</strong> ' . 
                 sprintf(__('Please configure your gateway credentials in the <a href="%s">payment settings</a>.', 'nass-payment-gateway'), 
                        admin_url('admin.php?page=wc-settings&tab=checkout&section=nass_gateway')) . '</p></div>';
        }
        
        // Check if SSL is enabled for production
        if ('production' === $gateway->environment && !is_ssl()) {
            echo '<div class="notice notice-error"><p><strong>' . __('NASS Payment Gateway', 'nass-payment-gateway') . ':</strong> ' . 
                 __('SSL certificate is required for production environment.', 'nass-payment-gateway') . '</p></div>';
        }
    }
}

// Initialize admin
new Nass_Admin();