<?php
/**
 * Plugin Name: NASS Payment Gateway
 * Plugin URI: https://kreatedev.com/nass-payment-gateway
 * Description: Integrate NASS E-Payment Gateway with WooCommerce for secure Mastercard and Visa card payments
 * Version: 1.0.0
 * Author: KREATE Technologies LLC
 * Author URI: https://kreatedev.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: nass-payment-gateway
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * WC requires at least: 5.0
 * WC tested up to: 9.5
 *
 * @package NassPaymentGateway
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('NASS_PLUGIN_VERSION', '1.0.0');
define('NASS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('NASS_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('NASS_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Declare HPOS compatibility
 */
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

/**
 * Main NASS Payment Gateway Plugin Class
 */
class NassPaymentGatewayPlugin {
    
    /**
     * Plugin instance
     * @var NassPaymentGatewayPlugin
     */
    private static $instance = null;
    
    /**
     * Get plugin instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'), 11);
        add_action('init', array($this, 'load_textdomain'));
        add_action('init', array($this, 'handle_debug_requests'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Handle debug requests
     */
    public function handle_debug_requests() {
        // Clear token cache
        if (isset($_GET['clear_nass_token']) && current_user_can('manage_options')) {
            delete_transient('nass_access_token_' . md5('info@epicstore-key.com'));
            wp_die('<h1>✅ NASS Token Cache Cleared!</h1><p><a href="/">← Go Home</a> | <a href="/checkout/">Go to Checkout</a></p>', 'Token Cleared');
        }
        
        // Show debug info
        if (isset($_GET['nass_debug']) && current_user_can('manage_options')) {
            $this->show_debug_info();
        }
    }
    
    /**
     * Show debug information
     */
    private function show_debug_info() {
        echo '<h1>NASS Gateway Debug Info</h1>';
        echo '<h2>Plugin Status</h2>';
        echo 'Plugin Version: ' . NASS_PLUGIN_VERSION . '<br>';
        echo 'WooCommerce Active: ' . (class_exists('WooCommerce') ? 'Yes' : 'No') . '<br>';
        echo 'Gateway Class Exists: ' . (class_exists('WC_Nass_Payment_Gateway') ? 'Yes' : 'No') . '<br>';
        
        if (class_exists('WooCommerce')) {
            $gateways = WC()->payment_gateways->get_available_payment_gateways();
            echo '<h2>Available Gateways (' . count($gateways) . ')</h2>';
            foreach ($gateways as $id => $gateway) {
                echo '- ' . $id . ': ' . $gateway->get_title() . '<br>';
            }
        }
        
        echo '<hr><p><a href="/">← Go Back</a></p>';
        exit;
    }
    
    /**
     * Initialize the plugin
     */
    public function init() {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }
        
        // Load plugin files
        $this->load_files();
        
        // Initialize gateway
        add_filter('woocommerce_payment_gateways', array($this, 'add_gateway_class'));
        
        // Add settings link to plugins page
        add_filter('plugin_action_links_' . NASS_PLUGIN_BASENAME, array($this, 'plugin_action_links'));
        
        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Add admin order display hooks
        $this->add_admin_order_hooks();
    }
    
    /**
     * Add admin order display hooks with HPOS compatibility
     */
    private function add_admin_order_hooks() {
        // HPOS compatible order details display
        add_action('woocommerce_admin_order_data_after_billing_address', array($this, 'display_payment_details_admin'), 10, 1);
        
        // Add payment ID column to orders list (HPOS compatible)
        add_filter('manage_woocommerce_page_wc-orders_columns', array($this, 'add_payment_id_column'));
        add_filter('manage_edit-shop_order_columns', array($this, 'add_payment_id_column'));
        
        // Display payment ID in orders list (HPOS compatible)
        add_action('manage_woocommerce_page_wc-orders_custom_column', array($this, 'display_payment_id_column'), 10, 2);
        add_action('manage_shop_order_posts_custom_column', array($this, 'display_payment_id_column_legacy'), 10, 2);
    }
    
    /**
     * Display payment details in admin order page
     */
    public function display_payment_details_admin($order) {
        $payment_method = $order->get_payment_method();
        
        if ('nass_gateway' !== $payment_method) {
            return;
        }
        
        $payment_id = $order->get_meta('_nass_payment_id');
        $rrn = $order->get_meta('_nass_rrn');
        $card_masked = $order->get_meta('_nass_card_masked');
        $payment_status = $order->get_meta('_nass_payment_status');
        
        if ($payment_id || $rrn) {
            echo '<h3>' . __('NASS Payment Details', 'nass-payment-gateway') . '</h3>';
            echo '<p>';
            
            if ($payment_id) {
                echo '<strong>' . __('Payment ID:', 'nass-payment-gateway') . '</strong> ' . esc_html($payment_id) . '<br>';
            }
            
            if ($rrn) {
                echo '<strong>' . __('Reference Number (RRN):', 'nass-payment-gateway') . '</strong> ' . esc_html($rrn) . '<br>';
            }
            
            if ($card_masked) {
                echo '<strong>' . __('Card:', 'nass-payment-gateway') . '</strong> ' . esc_html($card_masked) . '<br>';
            }
            
            if ($payment_status) {
                echo '<strong>' . __('Payment Status:', 'nass-payment-gateway') . '</strong> ' . esc_html($payment_status) . '<br>';
            }
            
            echo '</p>';
        }
    }
    
    /**
     * Add payment ID column
     */
    public function add_payment_id_column($columns) {
        $columns['nass_payment_id'] = __('NASS Payment ID', 'nass-payment-gateway');
        return $columns;
    }
    
    /**
     * Display payment ID in HPOS orders list
     */
    public function display_payment_id_column($column, $order) {
        if ('nass_payment_id' === $column) {
            if (is_numeric($order)) {
                $order = wc_get_order($order);
            }
            
            if ($order && 'nass_gateway' === $order->get_payment_method()) {
                $payment_id = $order->get_meta('_nass_payment_id');
                if ($payment_id) {
                    echo '<small>' . esc_html($payment_id) . '</small>';
                } else {
                    echo '-';
                }
            }
        }
    }
    
    /**
     * Display payment ID in legacy orders list
     */
    public function display_payment_id_column_legacy($column, $post_id) {
        $this->display_payment_id_column($column, $post_id);
    }
    
    /**
     * Load plugin text domain
     */
    public function load_textdomain() {
        load_plugin_textdomain('nass-payment-gateway', false, dirname(NASS_PLUGIN_BASENAME) . '/languages');
    }
    
    /**
     * Load required files
     */
    private function load_files() {
        require_once NASS_PLUGIN_PATH . 'includes/class-nass-logger.php';
        require_once NASS_PLUGIN_PATH . 'includes/class-nass-api.php';
        require_once NASS_PLUGIN_PATH . 'includes/class-nass-gateway.php';
        require_once NASS_PLUGIN_PATH . 'includes/admin/class-nass-admin.php';
    }
    
    /**
     * Add gateway to WooCommerce
     */
    public function add_gateway_class($gateways) {
        $gateways[] = 'WC_Nass_Payment_Gateway';
        return $gateways;
    }
    
    /**
     * Add settings link to plugin actions
     */
    public function plugin_action_links($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=wc-settings&tab=checkout&section=nass_gateway') . '">' . __('Settings', 'nass-payment-gateway') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueue_frontend_scripts() {
        if (is_checkout() || is_cart()) {
            wp_enqueue_style('nass-frontend-css', NASS_PLUGIN_URL . 'assets/css/frontend.css', array(), NASS_PLUGIN_VERSION);
            wp_enqueue_script('nass-checkout-js', NASS_PLUGIN_URL . 'assets/js/checkout.js', array('jquery'), NASS_PLUGIN_VERSION, true);
            
            // Localize script for AJAX
            wp_localize_script('nass-checkout-js', 'nass_checkout_params', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('nass_checkout_nonce'),
                'messages' => array(
                    'processing' => __('Processing payment...', 'nass-payment-gateway'),
                    'error' => __('Payment processing error. Please try again.', 'nass-payment-gateway'),
                    'invalid_card' => __('Please enter a valid card number.', 'nass-payment-gateway'),
                    'invalid_expiry' => __('Please enter a valid expiry date.', 'nass-payment-gateway'),
                    'invalid_cvc' => __('Please enter a valid security code.', 'nass-payment-gateway'),
                    'invalid_name' => __('Please enter the name on the card.', 'nass-payment-gateway')
                )
            ));
        }
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        if ('woocommerce_page_wc-settings' === $hook || strpos($hook, 'nass-payment-gateway') !== false) {
            wp_enqueue_style('nass-admin-css', NASS_PLUGIN_URL . 'assets/css/admin.css', array(), NASS_PLUGIN_VERSION);
            wp_enqueue_script('nass-admin-js', NASS_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), NASS_PLUGIN_VERSION, true);
            
            wp_localize_script('nass-admin-js', 'nass_admin_params', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('nass_admin_nonce')
            ));
        }
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Create database tables if needed
        $this->create_tables();
        
        // Set default options
        $this->set_default_options();
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up if needed
        flush_rewrite_rules();
    }
    
    /**
     * Create plugin database tables
     */
    private function create_tables() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'nass_payment_logs';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            order_id varchar(100) NOT NULL,
            transaction_id varchar(100),
            amount decimal(10,2) NOT NULL,
            currency varchar(3) NOT NULL,
            status varchar(50) NOT NULL,
            gateway_response text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY order_id (order_id),
            KEY transaction_id (transaction_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Set default plugin options
     */
    private function set_default_options() {
        $defaults = array(
            'enabled' => 'yes',
            'title' => 'NASS Payment Gateway',
            'description' => 'Pay securely with your Visa or Mastercard through NASS Payment Gateway',
            'environment' => 'sandbox',
            'terminal_id' => '00099619',
            'username' => 'info@epicstore-key.com',
            'password' => 'Epicstore@2025?',
            'gateway_url' => 'https://3dsecure.nass.iq/gateway/',
            'debug' => 'yes'
        );
        
        foreach ($defaults as $key => $value) {
            $option_key = 'woocommerce_nass_gateway_' . $key;
            if (!get_option($option_key)) {
                update_option($option_key, $value);
            }
        }
    }
    
    /**
     * WooCommerce missing notice
     */
    public function woocommerce_missing_notice() {
        echo '<div class="error"><p><strong>' . __('NASS Payment Gateway', 'nass-payment-gateway') . '</strong>: ' . 
             __('WooCommerce is required for this plugin to work. Please install and activate WooCommerce.', 'nass-payment-gateway') . '</p></div>';
    }
}

// Initialize the plugin
NassPaymentGatewayPlugin::get_instance();


// Force gateway registration for theme compatibility
add_filter('woocommerce_available_payment_gateways', function($gateways) {
    // Ensure NASS gateway is always included if available
    if (!isset($gateways['nass_gateway']) && class_exists('WC_Nass_Payment_Gateway')) {
        $nass_gateway = new WC_Nass_Payment_Gateway();
        if ($nass_gateway->is_available()) {
            $gateways['nass_gateway'] = $nass_gateway;
        }
    }
    return $gateways;
}, 999);

// ===== ADD THIS CODE BELOW =====

// Handle NASS payment returns on thank you page
add_action('woocommerce_before_thankyou', function($order_id) {
    // Check if this is a NASS return
    if (!isset($_GET['status']) || !isset($_GET['orderId'])) {
        return;
    }
    
    $order = wc_get_order($order_id);
    if (!$order || $order->get_payment_method() !== 'nass_gateway') {
        return;
    }
    
    // Get NASS parameters
    $status = sanitize_text_field($_GET['status']);
    $nass_order_id = sanitize_text_field($_GET['orderId']);
    $rrn = sanitize_text_field($_GET['rrn'] ?? '');
    $amount = sanitize_text_field($_GET['amount'] ?? '');
    
    // Log the return
    if (class_exists('Nass_Logger')) {
        $logger = new Nass_Logger(true);
        $logger->log("NASS return on thank you page - Order: {$order_id}, Status: {$status}, RRN: {$rrn}");
    }
    
    // Only process if order is still pending
    if ($order->get_status() === 'pending') {
        if ('success' === strtolower($status)) {
            // Payment successful
            $order->payment_complete($rrn ?: 'NASS-' . time());
            $order->update_status('processing');
            
            // Save payment details
            $order->update_meta_data('_nass_payment_id', $rrn);
            $order->update_meta_data('_nass_rrn', $rrn);
            $order->update_meta_data('_nass_payment_status', 'completed');
            $order->update_meta_data('_nass_payment_amount', $amount);
            $order->save();
            
            // Add order note
            $order->add_order_note(sprintf(
                __('NASS payment completed successfully. RRN: %s, Amount: %s', 'nass-payment-gateway'),
                $rrn,
                $amount
            ));
            
            // Show success message
            wc_add_notice(__('Payment received successfully!', 'nass-payment-gateway'), 'success');
        } else {
            // Payment failed
            $order->update_status('failed', sprintf(
                __('NASS payment failed. Status: %s, RRN: %s', 'nass-payment-gateway'),
                $status,
                $rrn ?: 'N/A'
            ));
            
            // Redirect to checkout with error
            wc_add_notice(__('Payment failed. Please try again.', 'nass-payment-gateway'), 'error');
            wp_redirect(wc_get_checkout_url());
            exit;
        }
    }
}, 5);



// Register custom API endpoint for NASS returns
// Register custom API endpoint for NASS returns
add_action('woocommerce_api_nass_payment_return', function() {
    // Get the full URL to handle the malformed format
    $full_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
    
    // Log the raw URL for debugging
    error_log('NASS Raw Return URL: ' . $full_url);
    
    // First try to get parameters normally
    $status = sanitize_text_field($_GET['status'] ?? '');
    $order_id = sanitize_text_field($_GET['orderId'] ?? $_GET['order_id'] ?? '');
    $rrn = sanitize_text_field($_GET['rrn'] ?? '');
    $amount = sanitize_text_field($_GET['amount'] ?? '');
    
    // If we don't have the parameters, parse the malformed URL
    if (empty($status) && strpos($full_url, '?status=') !== false) {
        // Extract the part after order_id
        preg_match('/order_id=(\d+)\?(.*)/', $full_url, $matches);
        
        if (isset($matches[2])) {
            // Parse the query string after the second ?
            parse_str($matches[2], $params);
            
            $status = sanitize_text_field($params['status'] ?? '');
            $order_id = sanitize_text_field($params['orderId'] ?? $matches[1]);
            $rrn = sanitize_text_field($params['rrn'] ?? '');
            $amount = sanitize_text_field($params['amount'] ?? '');
        }
    }
    
    // If still no order_id, try to extract it from the URL
    if (empty($order_id) && preg_match('/order_id=(\d+)/', $full_url, $matches)) {
        $order_id = $matches[1];
    }
    
    error_log('NASS Payment Return - Status: ' . $status . ', Order: ' . $order_id . ', RRN: ' . $rrn);
    
    if (!$order_id) {
        wc_add_notice(__('Invalid order ID.', 'nass-payment-gateway'), 'error');
        wp_redirect(wc_get_checkout_url());
        exit;
    }
    
    $order = wc_get_order($order_id);
    if (!$order) {
        wc_add_notice(__('Order not found.', 'nass-payment-gateway'), 'error');
        wp_redirect(wc_get_checkout_url());
        exit;
    }
    
    // Process based on status
    if ('success' === strtolower($status) || 'approved' === strtolower($status)) {
        // Payment successful
        if ($order->get_status() === 'pending') {
            // Complete the payment
            $order->payment_complete($rrn ?: 'NASS-' . time());
            $order->update_status('processing');
            
            // Save payment details
            $order->update_meta_data('_nass_payment_id', $rrn);
            $order->update_meta_data('_nass_rrn', $rrn);
            $order->update_meta_data('_nass_payment_status', 'completed');
            $order->update_meta_data('_nass_payment_amount', $amount);
            $order->save();
            
            // Add order note
            $order->add_order_note(sprintf(
                __('NASS payment completed successfully. RRN: %s, Amount: %s', 'nass-payment-gateway'),
                $rrn,
                $amount
            ));
            
            // Clear cart
            WC()->cart->empty_cart();
        }
        
        // Redirect to thank you page with parameters
        $thank_you_url = add_query_arg(array(
            'status' => $status,
            'orderId' => $order_id,
            'rrn' => $rrn,
            'amount' => $amount
        ), $order->get_checkout_order_received_url());
        
        wp_redirect($thank_you_url);
        exit;
        
    } else {
        // Payment failed
        if ($order->get_status() === 'pending') {
            // Cancel the order
            $order->update_status('cancelled', sprintf(
                __('NASS payment failed. Status: %s, RRN: %s', 'nass-payment-gateway'),
                $status,
                $rrn ?: 'N/A'
            ));
            
            // Restore cart
            WC()->cart->empty_cart();
            foreach ($order->get_items() as $item) {
                $product_id = $item->get_product_id();
                $quantity = $item->get_quantity();
                $variation_id = $item->get_variation_id();
                
                if ($variation_id) {
                    WC()->cart->add_to_cart($product_id, $quantity, $variation_id);
                } else {
                    WC()->cart->add_to_cart($product_id, $quantity);
                }
            }
        }
        
        // Add error notice
        wc_add_notice(__('Payment was declined. Please try again with a different card or payment method.', 'nass-payment-gateway'), 'error');
        
        // Redirect to checkout
        wp_redirect(wc_get_checkout_url());
        exit;
    }
});

// Prevent WooCommerce from auto-completing orders on thank you page
add_filter('woocommerce_payment_complete_order_status', function($status, $order_id, $order) {
    if ($order->get_payment_method() === 'nass_gateway' && $order->get_status() === 'pending') {
        // Don't auto-complete if we haven't received payment confirmation
        return 'pending';
    }
    return $status;
}, 10, 3);

// Hide order received page for cancelled NASS orders
add_action('template_redirect', function() {
    if (!is_wc_endpoint_url('order-received')) {
        return;
    }
    
    global $wp;
    $order_id = absint($wp->query_vars['order-received']);
    
    if ($order_id) {
        $order = wc_get_order($order_id);
        if ($order && $order->get_payment_method() === 'nass_gateway' && $order->get_status() === 'cancelled') {
            wp_redirect(wc_get_checkout_url());
            exit;
        }
    }
});



// Optional: Add this temporarily for debugging
add_action('init', function() {
    // Log NASS returns for debugging
    if (strpos($_SERVER['REQUEST_URI'], 'order-received') !== false && !empty($_GET)) {
        if (isset($_GET['status']) || isset($_GET['orderId'])) {
            error_log('NASS Return URL: ' . $_SERVER['REQUEST_URI']);
            error_log('NASS Return Params: ' . json_encode($_GET));
        }
    }
});


