<?php
/**
 * NASS Logger Class
 * 
 * Handles logging for NASS Payment Gateway
 *
 * @package NassPaymentGateway
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Nass_Logger Class
 */
class Nass_Logger {
    
    /**
     * Debug mode
     * @var bool
     */
    private $debug;
    
    /**
     * Log file path
     * @var string
     */
    private $log_file;
    
    /**
     * Constructor
     */
    public function __construct($debug = false) {
        $this->debug = $debug;
        $upload_dir = wp_upload_dir();
        $this->log_file = $upload_dir['basedir'] . '/nass-payment-logs.txt';
        
        // Ensure log directory exists
        if (!file_exists(dirname($this->log_file))) {
            wp_mkdir_p(dirname($this->log_file));
        }
    }
    
    /**
     * Log message
     */
    public function log($message, $level = 'info') {
        if (!$this->debug) {
            return;
        }
        
        $timestamp = current_time('Y-m-d H:i:s');
        $log_entry = sprintf("[%s] [%s] %s\n", $timestamp, strtoupper($level), $message);
        
        // Write to file
        file_put_contents($this->log_file, $log_entry, FILE_APPEND | LOCK_EX);
        
        // Also log to WordPress debug log if WP_DEBUG is enabled
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('NASS Gateway: ' . $message);
        }
    }
    
    /**
     * Log transaction details to database
     */
    public function log_transaction($order_id, $transaction_data) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'nass_payment_logs';
        
        $wpdb->insert(
            $table_name,
            array(
                'order_id' => $order_id,
                'transaction_id' => $transaction_data['rrn'] ?? $transaction_data['intRef'] ?? '',
                'amount' => $transaction_data['amount'] ?? 0,
                'currency' => $transaction_data['currency'] ?? '',
                'status' => $transaction_data['statusMsg'] ?? 'Unknown',
                'gateway_response' => json_encode($transaction_data),
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%f', '%s', '%s', '%s', '%s')
        );
        
        if ($this->debug) {
            $this->log('Transaction logged to database for order: ' . $order_id);
        }
    }
    
    /**
     * Get transaction logs
     */
    public function get_transaction_logs($order_id = null, $limit = 100) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'nass_payment_logs';
        
        $sql = "SELECT * FROM $table_name";
        $where_conditions = array();
        $prepare_values = array();
        
        if ($order_id) {
            $where_conditions[] = "order_id = %s";
            $prepare_values[] = $order_id;
        }
        
        if (!empty($where_conditions)) {
            $sql .= " WHERE " . implode(' AND ', $where_conditions);
        }
        
        $sql .= " ORDER BY created_at DESC LIMIT %d";
        $prepare_values[] = $limit;
        
        if (!empty($prepare_values)) {
            $sql = $wpdb->prepare($sql, $prepare_values);
        }
        
        return $wpdb->get_results($sql);
    }
    
    /**
     * Clear old logs
     */
    public function clear_old_logs($days = 30) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'nass_payment_logs';
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $deleted = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM $table_name WHERE created_at < %s",
                $cutoff_date
            )
        );
        
        if ($this->debug) {
            $this->log("Cleared {$deleted} old log entries older than {$days} days");
        }
        
        return $deleted;
    }
    
    /**
     * Get log file contents
     */
    public function get_log_contents($lines = 200) {
        if (!file_exists($this->log_file)) {
            return '';
        }
        
        $file = file($this->log_file);
        if (false === $file) {
            return '';
        }
        
        return implode('', array_slice($file, -$lines));
    }
    
    /**
     * Clear log file
     */
    public function clear_log_file() {
        if (file_exists($this->log_file)) {
            unlink($this->log_file);
            $this->log('Log file cleared');
        }
    }
    
    /**
     * Log API request
     */
    public function log_api_request($url, $data, $response) {
        if (!$this->debug) {
            return;
        }
        
        $log_data = array(
            'url' => $url,
            'request_data' => $data,
            'response' => $response
        );
        
        $this->log('API Request: ' . json_encode($log_data), 'api');
    }
    
    /**
     * Log API error
     */
    public function log_api_error($message, $context = array()) {
        $log_message = $message;
        if (!empty($context)) {
            $log_message .= ' Context: ' . json_encode($context);
        }
        
        $this->log($log_message, 'error');
    }
}