<?php
/**
 * NASS Payment Gateway Class
 * 
 * Handles the WooCommerce payment gateway integration
 *
 * @package NassPaymentGateway
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * WC_Nass_Payment_Gateway Class
 */
class WC_Nass_Payment_Gateway extends WC_Payment_Gateway {
    
    /**
     * API handler instance
     * @var Nass_API_Handler
     */
    private $api_handler;
    
    /**
     * Logger instance
     * @var Nass_Logger
     */
    private $logger;
    
    /**
     * Constructor
     */
    public function __construct() {
        // Basic gateway setup
        $this->id = 'nass_gateway';
        $this->icon = '';
        $this->has_fields = false; // Set to false for redirect gateway
        $this->method_title = __('NASS Payment Gateway', 'nass-payment-gateway');
        $this->method_description = __('Accept payments through NASS Payment Gateway using Visa and Mastercard', 'nass-payment-gateway');
        
        // Supported features
        $this->supports = array(
            'products'
        );
        
        // Load the settings
        $this->init_form_fields();
        $this->init_settings();
        
        // Define user set variables with defaults
        $this->title = $this->get_option('title', 'NASS Payment Gateway');
        $this->description = $this->get_option('description', 'Pay securely with your Visa or Mastercard through NASS Payment Gateway');
        $this->enabled = $this->get_option('enabled', 'yes');
        $this->environment = $this->get_option('environment', 'sandbox');
        $this->terminal_id = $this->get_option('terminal_id', '00099619');
        $this->username = $this->get_option('username', 'info@epicstore-key.com');
        $this->password = $this->get_option('password', 'Epicstore@2025?');
        $this->gateway_url = $this->get_option('gateway_url', 'https://3dsecure.nass.iq/gateway/');
        $this->debug = 'yes' === $this->get_option('debug', 'yes');
        
        // Initialize API handler and logger
        $this->api_handler = new Nass_API_Handler($this);
        $this->logger = new Nass_Logger($this->debug);
        
        // Hooks
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action('woocommerce_api_' . strtolower(get_class($this)), array($this, 'handle_callback'));
        
        // Handle return from NASS gateway
        add_action('init', array($this, 'handle_return'));
    }
    
    /**
     * Initialize Gateway Settings Form Fields
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'nass-payment-gateway'),
                'type' => 'checkbox',
                'label' => __('Enable NASS Payment Gateway', 'nass-payment-gateway'),
                'default' => 'no'
            ),
            'title' => array(
                'title' => __('Title', 'nass-payment-gateway'),
                'type' => 'text',
                'description' => __('Payment method title that customers will see during checkout.', 'nass-payment-gateway'),
                'default' => __('NASS Payment Gateway', 'nass-payment-gateway'),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __('Description', 'nass-payment-gateway'),
                'type' => 'textarea',
                'description' => __('Payment method description that customers will see during checkout.', 'nass-payment-gateway'),
                'default' => __('Pay securely with your Visa or Mastercard through NASS Payment Gateway', 'nass-payment-gateway'),
                'desc_tip' => true,
            ),
            'environment' => array(
                'title' => __('Environment', 'nass-payment-gateway'),
                'type' => 'select',
                'description' => __('Select the environment for transactions.', 'nass-payment-gateway'),
                'default' => 'sandbox',
                'desc_tip' => true,
                'options' => array(
                    'sandbox' => __('Sandbox (Testing)', 'nass-payment-gateway'),
                    'production' => __('Production (Live)', 'nass-payment-gateway')
                )
            ),
            'terminal_id' => array(
                'title' => __('Terminal ID', 'nass-payment-gateway'),
                'type' => 'text',
                'description' => __('Your NASS Payment Gateway Terminal ID', 'nass-payment-gateway'),
                'default' => '00099619',
                'desc_tip' => true,
            ),
            'username' => array(
                'title' => __('Username', 'nass-payment-gateway'),
                'type' => 'text',
                'description' => __('Your NASS Payment Gateway Username', 'nass-payment-gateway'),
                'default' => 'info@epicstore-key.com',
                'desc_tip' => true,
            ),
            'password' => array(
                'title' => __('Password', 'nass-payment-gateway'),
                'type' => 'password',
                'description' => __('Your NASS Payment Gateway Password', 'nass-payment-gateway'),
                'default' => 'Epicstore@2025?',
                'desc_tip' => true,
            ),
            'gateway_url' => array(
                'title' => __('Gateway URL', 'nass-payment-gateway'),
                'type' => 'text',
                'description' => __('NASS Payment Gateway URL', 'nass-payment-gateway'),
                'default' => 'https://3dsecure.nass.iq/gateway/',
                'desc_tip' => true,
            ),
            'debug' => array(
                'title' => __('Debug Log', 'nass-payment-gateway'),
                'type' => 'checkbox',
                'label' => __('Enable logging', 'nass-payment-gateway'),
                'default' => 'no',
                'description' => __('Log NASS events for debugging purposes.', 'nass-payment-gateway'),
                'desc_tip' => true,
            )
        );
    }
    
    /**
     * Check if gateway is available
     */
    public function is_available() {
        if ('yes' !== $this->enabled) {
            return false;
        }
        
        // Check required settings
        if (empty($this->terminal_id) || empty($this->username) || empty($this->password)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Payment form on checkout page
     */
    public function payment_fields() {
        // Debug
        if ($this->debug) {
            $this->logger->log('Displaying payment fields');
        }
        
        if ($this->description) {
            echo '<div class="nass-payment-description">';
            echo '<p>' . wp_kses_post($this->description) . '</p>';
            echo '</div>';
        }
        
        // Simple redirect notice - no input fields needed
        echo '<div class="nass-payment-notice" style="background: #e3f2fd; border: 1px solid #2196f3; padding: 15px; border-radius: 5px; margin: 10px 0;">';
        echo '<p><strong>' . __('Secure Payment Process:', 'nass-payment-gateway') . '</strong></p>';
        echo '<p>' . __('Click "Place Order" and you will be redirected to enter your card details securely on the NASS payment page.', 'nass-payment-gateway') . '</p>';
        echo '<p><small>' . __('Accepted: Visa, Mastercard | 3D Secure Protected', 'nass-payment-gateway') . '</small></p>';
        echo '</div>';
    }
    
    /**
     * Process the payment
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            return array(
                'result' => 'failure',
                'messages' => __('Invalid order.', 'nass-payment-gateway')
            );
        }
        
        try {
            // Log payment attempt
            $this->logger->log('Processing payment for order: ' . $order_id);
            
            // Build proper return URL - use a custom endpoint
            // Build proper return URL - use a custom endpoint
            $return_url = add_query_arg(array(
                'wc-api' => 'nass_payment_return',
                'order_id' => $order_id
            ), home_url('/'));
            
            // Prepare transaction data according to documentation
            $transaction_data = array(
                'orderId' => strval($order_id), // Use order ID directly as string
                'orderDesc' => $this->get_order_description($order),
                'amount' => number_format($order->get_total(), 2, '.', ''), // Format amount properly
                'currency' => $this->get_currency_code($order->get_currency()),
                'transactionType' => '1',
                'backRef' => $return_url, // Use custom return URL
                'notifyUrl' => WC()->api_request_url(get_class($this))
            );
            
            $this->logger->log('Transaction data: ' . json_encode($transaction_data));
            
            // Create transaction through API
            $response = $this->api_handler->create_transaction($transaction_data);
            
            if (!$response || !isset($response['success']) || !$response['success']) {
                $error_msg = isset($response['message']) ? $response['message'] : 'Unknown error';
                throw new Exception(__('Failed to create transaction: ', 'nass-payment-gateway') . $error_msg);
            }
            
            // Verify we have the URL
            if (!isset($response['data']['url']) || empty($response['data']['url'])) {
                throw new Exception(__('No payment URL received from gateway', 'nass-payment-gateway'));
            }
            
            $payment_url = $response['data']['url'];
            
            // Store transaction data in order meta
            $order->update_meta_data('_nass_transaction_data', $response);
            $order->update_meta_data('_nass_transaction_id', $response['data']['transactionParams']['NONCE'] ?? '');
            $order->update_meta_data('_nass_order_id', $order_id);
            $order->save();
            
            // Mark order as pending payment
            $order->update_status('pending', __('Awaiting NASS payment', 'nass-payment-gateway'));
            
            $this->logger->log('Redirecting to payment URL: ' . $payment_url);
            
            // Return success with redirect URL (WooCommerce will handle the redirect in same tab)
            return array(
                'result' => 'success',
                'redirect' => $payment_url
            );
            
        } catch (Exception $e) {
            $this->logger->log('Payment processing error: ' . $e->getMessage());
            wc_add_notice(__('Payment error: ', 'nass-payment-gateway') . $e->getMessage(), 'error');
            return array(
                'result' => 'failure'
            );
        }
    }
    
    /**
     * Handle payment callback from NASS
     */
    public function handle_callback() {
        $this->logger->log('Callback received from NASS');
        
        $input = file_get_contents('php://input');
        $callback_data = json_decode($input, true);
        
        if (!$callback_data) {
            $this->logger->log('Invalid callback data received');
            wp_die('Invalid callback data', 'NASS Callback', array('response' => 400));
        }
        
        $this->logger->log('Callback data: ' . print_r($callback_data, true));
        
        // Find order by order ID
        $order_id = $callback_data['orderId'] ?? '';
        if (empty($order_id)) {
            $this->logger->log('No order ID in callback data');
            wp_die('No order ID provided', 'NASS Callback', array('response' => 400));
        }
        
        // Get order - HPOS compatible
        $order = wc_get_order($order_id);
        
        if (!$order) {
            // Try to find by order number using HPOS-compatible method
            $orders = wc_get_orders(array(
                'limit' => 1,
                'orderby' => 'date',
                'order' => 'DESC',
                'meta_query' => array(
                    array(
                        'key' => '_order_number',
                        'value' => $order_id,
                        'compare' => '='
                    )
                )
            ));
            
            $order = !empty($orders) ? $orders[0] : null;
        }
        
        if (!$order) {
            $this->logger->log('Order not found: ' . $order_id);
            wp_die('Order not found', 'NASS Callback', array('response' => 404));
        }
        
        // Process callback based on response code
        $response_code = $callback_data['responseCode'] ?? '';
        $status_msg = $callback_data['statusMsg'] ?? '';
        $transaction_id = $callback_data['rrn'] ?? $callback_data['intRef'] ?? '';
        
        if ('00' === $response_code && 'Approved' === $status_msg) {
            // Payment successful
            $this->complete_order_payment($order, $transaction_id, $callback_data);
            $this->logger->log('Payment completed for order: ' . $order->get_id());
        } else {
            // Payment failed
            $order->update_status('failed', sprintf(__('NASS payment failed. Response: %s', 'nass-payment-gateway'), $status_msg));
            $this->logger->log('Payment failed for order: ' . $order->get_id() . '. Response: ' . $status_msg);
        }
        
        // Log the transaction
        $this->logger->log_transaction($order->get_id(), $callback_data);
        
        wp_die('OK', 'NASS Callback', array('response' => 200));
    }
    
   /**
 * Handle return from NASS gateway - FIXED VERSION
 */
public function handle_return() {
    // Don't process here if it's being handled by the API endpoint
    if (isset($_GET['wc-api']) && $_GET['wc-api'] === 'nass_payment_return') {
        return; // Let the API handler in main plugin file handle this
    }
    
    // Only process if we have NASS parameters but NOT the wc-api parameter
    if (!isset($_GET['status']) || !isset($_GET['orderId'])) {
        return;
    }
    
    // This code will only run if NASS returns directly without using our API endpoint
    // (which shouldn't happen based on your setup, but keeping as fallback)
    
    $order_id = sanitize_text_field($_GET['orderId']);
    $status = sanitize_text_field($_GET['status'] ?? '');
    $rrn = sanitize_text_field($_GET['rrn'] ?? '');
    $amount = sanitize_text_field($_GET['amount'] ?? '');
    
    $this->logger->log("Direct return from NASS - Order: {$order_id}, Status: {$status}, RRN: {$rrn}, Amount: {$amount}");
    
    // Find order
    $order = wc_get_order($order_id);
    
    if (!$order) {
        $this->logger->log('Order not found: ' . $order_id);
        wc_add_notice(__('Order not found. Please contact support.', 'nass-payment-gateway'), 'error');
        wp_redirect(wc_get_checkout_url());
        exit;
    }
    
    // Process based on status
    if ('success' === strtolower($status) || 'approved' === strtolower($status)) {
        // Only update if order is not already completed
        if (!in_array($order->get_status(), array('processing', 'completed'))) {
            // Build payment data
            $payment_data = array(
                'status' => $status,
                'orderId' => $order_id,
                'rrn' => $rrn,
                'amount' => $amount
            );
            
            // Complete the order payment
            $this->complete_order_payment($order, $rrn ?: 'NASS-' . time(), $payment_data);
        }
        
        // Redirect to proper thank you page with parameters
        $thank_you_url = add_query_arg(array(
            'status' => $status,
            'orderId' => $order_id,
            'rrn' => $rrn,
            'amount' => $amount
        ), $order->get_checkout_order_received_url());
        
        wp_redirect($thank_you_url);
        exit;
    } else {
        // Payment failed - cancel the order
        $order->update_status('cancelled', sprintf(
            __('NASS payment failed. Status: %s, RRN: %s', 'nass-payment-gateway'), 
            $status, 
            $rrn ?: 'N/A'
        ));
        
        // Add detailed note
        $order->add_order_note(sprintf(
            __('Payment failed at NASS gateway. Status: %s, RRN: %s, Amount: %s. Order cancelled.', 'nass-payment-gateway'),
            $status,
            $rrn ?: 'N/A',
            $amount
        ));
        
        // Clear cart session to allow retry
        WC()->cart->empty_cart();
        
        // Restore cart items
        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();
            $quantity = $item->get_quantity();
            $variation_id = $item->get_variation_id();
            
            if ($variation_id) {
                WC()->cart->add_to_cart($product_id, $quantity, $variation_id);
            } else {
                WC()->cart->add_to_cart($product_id, $quantity);
            }
        }
        
        // Set error message
        wc_add_notice(__('Payment was declined. Please try again with a different card or payment method.', 'nass-payment-gateway'), 'error');
        
        // Redirect to checkout
        wp_redirect(wc_get_checkout_url());
        exit;
    }
}
    
    /**
     * Complete order payment - NEW METHOD
     */
    private function complete_order_payment($order, $transaction_id, $payment_data) {
        // Mark payment complete
        $order->payment_complete($transaction_id);
        
        // Update order status to processing
        $order->update_status('processing');
        
        // Save payment metadata
        $order->update_meta_data('_nass_payment_id', $transaction_id);
        $order->update_meta_data('_nass_rrn', $payment_data['rrn'] ?? '');
        $order->update_meta_data('_nass_payment_status', 'completed');
        $order->update_meta_data('_nass_payment_data', $payment_data);
        
        // Save card info if available (masked)
        if (isset($payment_data['card'])) {
            $order->update_meta_data('_nass_card_masked', $payment_data['card']);
        }
        
        $order->save();
        
        // Add detailed order note
        $note_parts = array(
            sprintf(__('NASS payment completed successfully.', 'nass-payment-gateway')),
            sprintf(__('Payment ID: %s', 'nass-payment-gateway'), $transaction_id)
        );
        
        if (isset($payment_data['rrn'])) {
            $note_parts[] = sprintf(__('Reference Number (RRN): %s', 'nass-payment-gateway'), $payment_data['rrn']);
        }
        
        if (isset($payment_data['card'])) {
            $note_parts[] = sprintf(__('Card: %s', 'nass-payment-gateway'), $payment_data['card']);
        }
        
        if (isset($payment_data['amount'])) {
            $note_parts[] = sprintf(__('Amount: %s %s', 'nass-payment-gateway'), $payment_data['amount'], $order->get_currency());
        }
        
        $order->add_order_note(implode("\n", $note_parts));
        
        $this->logger->log('Payment completed for order: ' . $order->get_id() . ', Payment ID: ' . $transaction_id);
    }
    
    /**
     * Thank you page
     */
    public function thankyou_page($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) return;
        
        $payment_id = $order->get_meta('_nass_payment_id');
        $rrn = $order->get_meta('_nass_rrn');
        
        if ($payment_id) {
            echo '<p><strong>' . __('Payment Information:', 'nass-payment-gateway') . '</strong></p>';
            echo '<ul>';
            echo '<li>' . sprintf(__('Payment ID: %s', 'nass-payment-gateway'), esc_html($payment_id)) . '</li>';
            if ($rrn) {
                echo '<li>' . sprintf(__('Reference Number: %s', 'nass-payment-gateway'), esc_html($rrn)) . '</li>';
            }
            echo '</ul>';
        }
    }
    
    /**
     * Process refund
     */
    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            return new WP_Error('invalid_order', __('Invalid order for refund.', 'nass-payment-gateway'));
        }
        
        $transaction_id = $order->get_meta('_nass_payment_id');
        
        if (empty($transaction_id)) {
            return new WP_Error('no_transaction_id', __('No payment ID found for refund.', 'nass-payment-gateway'));
        }
        
        try {
            // Note: NASS gateway refund implementation would go here
            // For now, we'll just log the refund request
            $this->logger->log('Refund requested for order: ' . $order_id . ', amount: ' . $amount . ', reason: ' . $reason);
            
            return new WP_Error('refund_not_implemented', __('Refunds are not yet implemented for NASS gateway.', 'nass-payment-gateway'));
            
        } catch (Exception $e) {
            return new WP_Error('refund_error', $e->getMessage());
        }
    }
    
    /**
     * Get order description
     */
    private function get_order_description($order) {
        $items = array();
        foreach ($order->get_items() as $item) {
            $items[] = $item->get_name();
        }
        
        $description = implode(', ', array_slice($items, 0, 3));
        if (count($items) > 3) {
            $description .= '...';
        }
        
        return $description ?: sprintf(__('Order #%s', 'nass-payment-gateway'), $order->get_order_number());
    }
    
    /**
     * Get currency code for NASS
     */
    private function get_currency_code($currency) {
        $currency_codes = array(
            'USD' => '840',
            'EUR' => '978',
            'IQD' => '368',
            'GBP' => '826'
        );
        
        return $currency_codes[$currency] ?? '368'; // Default to IQD
    }
    
    /**
     * Load payment scripts
     */
    public function payment_scripts() {
        if (!is_cart() && !is_checkout() && !isset($_GET['pay_for_order'])) {
            return;
        }
        
        if ('no' === $this->enabled) {
            return;
        }
        
        // Scripts are already loaded in main plugin file
    }
}