<?php
/**
 * NASS API Handler Class
 * 
 * Handles all API communications with NASS Payment Gateway
 *
 * @package NassPaymentGateway
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Nass_API_Handler Class
 */
class Nass_API_Handler {
    
    /**
     * Gateway instance
     * @var WC_Nass_Payment_Gateway
     */
    private $gateway;
    
    /**
     * API base URL
     * @var string
     */
    private $base_url;
    
    /**
     * Access token
     * @var string
     */
    private $access_token;
    
    /**
     * Logger instance
     * @var Nass_Logger
     */
    private $logger;
    
    /**
     * Constructor
     */
    public function __construct($gateway) {
        $this->gateway = $gateway;
        $this->logger = new Nass_Logger($gateway->debug);
        $this->set_base_url();
    }
    
    /**
     * Set base URL based on environment
     */
    private function set_base_url() {
        if ('production' === $this->gateway->environment) {
            // Production URL would be provided by NASS
            $this->base_url = 'https://gateway.nass.iq/'; // Placeholder - replace with actual production URL
        } else {
            // UAT Environment
            $this->base_url = 'https://uat-gateway.nass.iq:9746/';
        }
    }
    
    /**
     * Authenticate with NASS API
     */
    public function authenticate() {
        $auth_data = array(
            'username' => $this->gateway->username,
            'password' => $this->gateway->password
        );
        
        $this->logger->log('Attempting authentication with NASS API');
        
        try {
            $response = $this->make_request('auth/merchant/login', $auth_data, 'POST');
            
            // Handle both direct token response and nested response
            if ($response && (isset($response['access_token']) || (isset($response['data']['access_token'])))) {
                $this->access_token = $response['access_token'] ?? $response['data']['access_token'];
                
                // Store token with expiration (assuming 1 hour)
                set_transient('nass_access_token_' . md5($this->gateway->username), $this->access_token, HOUR_IN_SECONDS);
                
                $this->logger->log('Authentication successful - Token received');
                return true;
            }
            
            $this->logger->log('Authentication failed - Response: ' . json_encode($response));
            return false;
            
        } catch (Exception $e) {
            $this->logger->log('Authentication error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get access token (with caching)
     */
    private function get_access_token() {
        // Check for cached token first
        $cached_token = get_transient('nass_access_token_' . md5($this->gateway->username));
        
        if ($cached_token) {
            $this->access_token = $cached_token;
            return $this->access_token;
        }
        
        // Authenticate to get new token
        if ($this->authenticate()) {
            return $this->access_token;
        }
        
        return false;
    }
    
    /**
     * Create transaction
     */
    public function create_transaction($transaction_data) {
        // Ensure we have an access token
        $token = $this->get_access_token();
        if (!$token) {
            throw new Exception(__('Failed to authenticate with NASS gateway', 'nass-payment-gateway'));
        }
        
        // Ensure all required fields are present and properly formatted
        $request_data = array(
            'orderId' => strval($transaction_data['orderId']),
            'orderDesc' => substr($transaction_data['orderDesc'], 0, 50),
            'amount' => number_format(floatval($transaction_data['amount']), 2, '.', ''),
            'currency' => strval($transaction_data['currency']),
            'transactionType' => '1',
            'backRef' => $transaction_data['backRef'],
            'notifyUrl' => $transaction_data['notifyUrl']
        );

        // Ensure URLs are not HTML-encoded
        $request_data['backRef'] = html_entity_decode($request_data['backRef'], ENT_QUOTES | ENT_HTML5);
        $request_data['notifyUrl'] = html_entity_decode($request_data['notifyUrl'], ENT_QUOTES | ENT_HTML5);
        
        $this->logger->log('Creating transaction with data: ' . json_encode($request_data));
        
        // Make transaction request
        $response = $this->make_request('transaction', $request_data, 'POST', true);
        
        if (!$response) {
            throw new Exception(__('No response from NASS gateway', 'nass-payment-gateway'));
        }
        
        // Check for errors in response
        if (isset($response['success']) && !$response['success']) {
            $error_msg = $response['message'] ?? __('Transaction creation failed', 'nass-payment-gateway');
            throw new Exception($error_msg);
        }
        
        // Verify we have the payment URL
        if (!isset($response['data']['url']) || empty($response['data']['url'])) {
            $this->logger->log('Invalid response structure: ' . json_encode($response));
            throw new Exception(__('Invalid response from payment gateway - no payment URL', 'nass-payment-gateway'));
        }
        
        $this->logger->log('Transaction created successfully');
        return $response;
    }
    
    /**
     * Check transaction status
     */
    public function check_transaction_status($order_id) {
        // Ensure we have an access token
        $token = $this->get_access_token();
        if (!$token) {
            $this->logger->log('Cannot check transaction status - authentication failed');
            return false;
        }
        
        $endpoint = 'transaction/' . $order_id . '/checkStatus';
        $this->logger->log('Checking transaction status for order: ' . $order_id);
        
        $response = $this->make_request($endpoint, null, 'GET', true);
        
        if ($response) {
            $this->logger->log('Transaction status retrieved: ' . json_encode($response));
        } else {
            $this->logger->log('Failed to retrieve transaction status');
        }
        
        return $response;
    }
    
    /**
     * Enhanced make_request with better error handling
     */
    private function make_request($endpoint, $data = null, $method = 'GET', $authenticated = false) {
        $url = $this->base_url . $endpoint;
        
        $args = array(
            'timeout' => 45,
            'method' => $method,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'User-Agent' => 'WordPress/' . get_bloginfo('version') . '; ' . home_url(),
                'Cache-Control' => 'no-cache'
            ),
            'sslverify' => false, // For UAT environment
            'blocking' => true
        );
        
        // Add authorization header if authenticated request
        if ($authenticated && $this->access_token) {
            $args['headers']['Authorization'] = 'Bearer ' . $this->access_token;
        }
        
        // Add request body for POST requests
        if ('POST' === $method && $data) {
            $args['body'] = json_encode($data);
        }
        
        // Log request for debugging
        $this->logger->log('API Request to ' . $url);
        if ($data) {
            $this->logger->log('Request data: ' . json_encode($data));
        }
        
        // Make the request
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->logger->log('API request failed: ' . $error_message);
            throw new Exception('API request failed: ' . $error_message);
        }
        
        $body = wp_remote_retrieve_body($response);
        $http_code = wp_remote_retrieve_response_code($response);
        
        $this->logger->log('API Response Code: ' . $http_code);
        $this->logger->log('API Response Body: ' . $body);
        
        // Parse JSON response
        $parsed_response = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->logger->log('JSON parse error: ' . json_last_error_msg());
            throw new Exception('Invalid JSON response from gateway');
        }
        
        // Handle HTTP errors
        if ($http_code >= 400) {
            if (401 === $http_code && $authenticated) {
                // Clear cached token and try once more
                delete_transient('nass_access_token_' . md5($this->gateway->username));
                $this->access_token = null;
                
                // Get fresh token and retry
                $fresh_token = $this->get_access_token();
                if ($fresh_token) {
                    $args['headers']['Authorization'] = 'Bearer ' . $fresh_token;
                    $retry_response = wp_remote_request($url, $args);
                    
                    if (!is_wp_error($retry_response)) {
                        $retry_body = wp_remote_retrieve_body($retry_response);
                        $retry_code = wp_remote_retrieve_response_code($retry_response);
                        
                        if ($retry_code < 400) {
                            return json_decode($retry_body, true);
                        }
                    }
                }
            }
            
            $error_msg = isset($parsed_response['message']) ? $parsed_response['message'] : 'HTTP Error ' . $http_code;
            throw new Exception($error_msg);
        }
        
        return $parsed_response;
    }
    
    /**
     * Process direct payment (for Direct API Integration)
     */
    public function process_direct_payment($payment_data) {
        // Direct API integration URL
        $direct_url = 'https://3dsecure.nass.iq/cgi-bin/cgi_json';
        
        // Prepare direct payment request according to documentation
        $request_data = array(
            'card' => $payment_data['card'],
            'expMonth' => $payment_data['expMonth'],
            'expYear' => $payment_data['expYear'],
            'currency' => $payment_data['currency'],
            'name' => $payment_data['name'],
            'trtype' => '1', // Purchase transaction
            'terminal' => $this->gateway->terminal_id,
            'desc' => $payment_data['desc'],
            'cvc2_rc' => $payment_data['cvc2_rc'] ?? '1', // 1 = CVC2 is present
            'orderId' => $payment_data['orderId'],
            'timestamp' => $payment_data['timestamp'],
            'nonce' => $payment_data['nonce'],
            'p_sign' => $payment_data['p_sign'],
            'amount' => $payment_data['amount'],
            'cvc2' => $payment_data['cvc2'],
            'backRefUrl' => $payment_data['backRefUrl']
        );
        
        $this->logger->log('Processing direct payment for order: ' . $payment_data['orderId']);
        
        // Make direct payment request
        $response = wp_remote_post($direct_url, array(
            'timeout' => 45,
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'WordPress/' . get_bloginfo('version') . '; ' . home_url()
            ),
            'body' => json_encode($request_data),
            'sslverify' => false // For UAT - set to true in production
        ));
        
        if (is_wp_error($response)) {
            $this->logger->log('Direct payment error: ' . $response->get_error_message());
            throw new Exception($response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $http_code = wp_remote_retrieve_response_code($response);
        
        $this->logger->log('Direct payment response code: ' . $http_code);
        
        if (200 !== $http_code) {
            $this->logger->log('Direct payment HTTP error: ' . $http_code);
            throw new Exception('HTTP Error: ' . $http_code);
        }
        
        return $body; // Returns HTML response
    }
    
    /**
     * Validate signature (for webhook verification)
     */
    public function validate_signature($data, $signature) {
        // Implementation depends on NASS signature algorithm
        // This is a placeholder for the actual signature validation
        
        $this->logger->log('Validating signature for webhook data');
        
        // For now, return true - replace with actual validation
        return true;
    }
    
    /**
     * Generate transaction signature
     */
    public function generate_signature($data) {
        // Implementation depends on NASS signature requirements
        $string_to_sign = '';
        foreach ($data as $key => $value) {
            $string_to_sign .= $key . '=' . $value . '&';
        }
        $string_to_sign = rtrim($string_to_sign, '&');
        
        $signature = hash('sha256', $string_to_sign);
        $this->logger->log('Generated signature for transaction data');
        
        return $signature;
    }
    
    /**
     * Generate nonce for requests
     */
    public function generate_nonce() {
        return bin2hex(random_bytes(16));
    }
    
    /**
     * Generate timestamp
     */
    public function generate_timestamp() {
        return date('YmdHis');
    }
}