/**
 * NASS Payment Gateway Frontend JavaScript
 */

jQuery(document).ready(function($) {
    'use strict';
    
    // NASS Payment Gateway object
    var NassPaymentGateway = {
        
        /**
         * Initialize the payment form
         */
        init: function() {
            this.bindEvents();
            this.initializeForm();
            this.initializeCardFields();
        },
        
        /**
         * Initialize card input fields
         */
        initializeCardFields: function() {
            // Format card number as user types
            $('#nass_card_number').on('input', function() {
                var value = $(this).val().replace(/\s+/g, '').replace(/[^0-9]/gi, '');
                var formattedValue = value.match(/.{1,4}/g)?.join(' ') || value;
                $(this).val(formattedValue);
            });
            
            // Format expiry date as user types
            $('#nass_card_expiry').on('input', function() {
                var value = $(this).val().replace(/\s+/g, '').replace(/[^0-9]/gi, '');
                if (value.length >= 2) {
                    value = value.substring(0, 2) + '/' + value.substring(2, 4);
                }
                $(this).val(value);
            });
            
            // Only allow numbers in CVV
            $('#nass_card_cvv').on('input', function() {
                $(this).val($(this).val().replace(/[^0-9]/g, ''));
            });
            
            // Remove error styling on input
            $('.nass-card-fields input').on('input', function() {
                $(this).removeClass('error');
                $(this).parent().find('.field-feedback').remove();
            });
        },
        
        /**
         * Bind events
         */
        bindEvents: function() {
            var self = this;
            
            // Payment method selection
            $(document).on('change', 'input[name="payment_method"]', function() {
                if ($(this).val() === 'nass_gateway') {
                    self.showPaymentForm();
                } else {
                    self.hidePaymentForm();
                }
            });
            
            // Form validation on submit
            $('form.checkout').on('checkout_place_order_nass_gateway', function() {
                return self.validateAndProcess();
            });
            
            // Handle checkout errors
            $(document.body).on('checkout_error', function() {
                self.hideProcessing();
            });
            
            // Test card auto-fill on click
            $(document).on('click', '.test-cards p', function() {
                self.fillTestCard($(this));
            });
        },
        
        /**
         * Validate form and process payment
         */
        validateAndProcess: function() {
            // Clear any existing errors
            $('.nass-payment-form .woocommerce-error').remove();
            $('.nass-card-fields input').removeClass('error');
            $('.field-feedback').remove();
            
            var isValid = true;
            var errors = [];
            
            // Validate card number
            var cardNumber = $('#nass_card_number').val().replace(/\s+/g, '');
            if (!this.validateCardNumber(cardNumber)) {
                isValid = false;
                errors.push('Please enter a valid card number');
                $('#nass_card_number').addClass('error');
                this.showFieldError('#nass_card_number', 'Invalid card number');
            }
            
            // Validate cardholder name
            var cardName = $('#nass_card_name').val().trim();
            if (cardName === '' || cardName.length < 3) {
                isValid = false;
                errors.push('Please enter cardholder name');
                $('#nass_card_name').addClass('error');
                this.showFieldError('#nass_card_name', 'Required field');
            }
            
            // Validate expiry
            var expiry = $('#nass_card_expiry').val();
            if (!this.validateExpiry(expiry)) {
                isValid = false;
                errors.push('Please enter valid expiry date (MM/YY)');
                $('#nass_card_expiry').addClass('error');
                this.showFieldError('#nass_card_expiry', 'Invalid date');
            }
            
            // Validate CVV
            var cvv = $('#nass_card_cvv').val();
            if (!this.validateCVV(cvv)) {
                isValid = false;
                errors.push('Please enter valid CVV');
                $('#nass_card_cvv').addClass('error');
                this.showFieldError('#nass_card_cvv', 'Invalid CVV');
            }
            
            if (!isValid) {
                this.showErrors(errors);
                return false;
            }
            
            // Show processing state
            this.showProcessing();
            return true;
        },
        
        /**
         * Validate card number using Luhn algorithm
         */
        validateCardNumber: function(cardNumber) {
            if (cardNumber.length < 13 || cardNumber.length > 19) {
                return false;
            }
            
            // Luhn algorithm
            var sum = 0;
            var isEven = false;
            
            for (var i = cardNumber.length - 1; i >= 0; i--) {
                var digit = parseInt(cardNumber[i]);
                
                if (isEven) {
                    digit *= 2;
                    if (digit > 9) {
                        digit -= 9;
                    }
                }
                
                sum += digit;
                isEven = !isEven;
            }
            
            return sum % 10 === 0;
        },
        
        /**
         * Validate expiry date
         */
        validateExpiry: function(expiry) {
            if (!expiry.match(/^(0[1-9]|1[0-2])\/\d{2}$/)) {
                return false;
            }
            
            var parts = expiry.split('/');
            var month = parseInt(parts[0]);
            var year = parseInt('20' + parts[1]);
            
            var now = new Date();
            var currentMonth = now.getMonth() + 1;
            var currentYear = now.getFullYear();
            
            if (year < currentYear || (year === currentYear && month < currentMonth)) {
                return false;
            }
            
            return true;
        },
        
        /**
         * Validate CVV
         */
        validateCVV: function(cvv) {
            return cvv.length >= 3 && cvv.length <= 4 && /^\d+$/.test(cvv);
        },
        
        /**
         * Show field-specific error
         */
        showFieldError: function(field, message) {
            var $field = $(field);
            var $feedback = $('<span class="field-feedback error">' + message + '</span>');
            $field.after($feedback);
        },
        
        /**
         * Fill test card details
         */
        fillTestCard: function($element) {
            var text = $element.text();
            var match = text.match(/:\s*(.+)/);
            
            if (match) {
                var value = match[1];
                
                if (text.includes('Visa') || text.includes('Mastercard')) {
                    $('#nass_card_number').val(value).trigger('input');
                    $('#nass_card_name').val('Test User');
                } else if (text.includes('Expiry')) {
                    var expiryOptions = value.split(' or ');
                    $('#nass_card_expiry').val(expiryOptions[0]);
                } else if (text.includes('CVV')) {
                    $('#nass_card_cvv').val(value);
                }
                
                this.showCopySuccess($element);
            }
        },
        
        /**
         * Show/hide payment form methods
         */
        showPaymentForm: function() {
            $('.nass-payment-form').slideDown(300);
        },
        
        hidePaymentForm: function() {
            $('.nass-payment-form').slideUp(300);
            this.hideProcessing();
        },
        
        /**
         * Initialize form on page load
         */
        initializeForm: function() {
            if ($('input[name="payment_method"]:checked').val() === 'nass_gateway') {
                this.showPaymentForm();
            } else {
                this.hidePaymentForm();
            }
        },
        
        /**
         * Show validation errors
         */
        showErrors: function(errors) {
            if (errors.length === 0) return;
            
            var errorHtml = '<div class="woocommerce-error">' + errors.join('<br>') + '</div>';
            $('.nass-payment-form').prepend(errorHtml);
            
            $('html, body').animate({
                scrollTop: $('.nass-payment-form').offset().top - 100
            }, 500);
        },
        
        /**
         * Show processing state
         */
        showProcessing: function() {
            $('.nass-loading-overlay').show();
            $('#place_order').prop('disabled', true);
            $('.nass-payment-form').addClass('processing');
        },
        
        /**
         * Hide processing state
         */
        hideProcessing: function() {
            $('.nass-loading-overlay').hide();
            $('#place_order').prop('disabled', false);
            $('.nass-payment-form').removeClass('processing');
        },
        
        /**
         * Show copy success message
         */
        showCopySuccess: function($element) {
            var $msg = $('<span class="copied-msg">Filled!</span>');
            $msg.css({
                'color': '#27ae60',
                'font-weight': 'bold',
                'margin-left': '10px',
                'font-size': '12px'
            });
            
            $element.append($msg);
            setTimeout(function() {
                $msg.fadeOut(function() {
                    $msg.remove();
                });
            }, 2000);
        }
    };
    
    // Initialize when DOM is ready
    NassPaymentGateway.init();
    
    // Handle fragment refresh
    $(document.body).on('updated_checkout', function() {
        NassPaymentGateway.initializeForm();
        NassPaymentGateway.initializeCardFields();
    });
});