/**
 * NASS Payment Gateway Admin JavaScript
 */

jQuery(document).ready(function($) {
    'use strict';
    
    // NASS Admin object
    var NassAdmin = {
        
        /**
         * Initialize admin functionality
         */
        init: function() {
            this.bindEvents();
            this.initModal();
            this.initDashboard();
        },
        
        /**
         * Bind admin events
         */
        bindEvents: function() {
            var self = this;
            
            // Test connection button
            $(document).on('click', '#test-connection', function(e) {
                e.preventDefault();
                self.testConnection();
            });
            
            // Clear logs button
            $(document).on('click', '#clear-logs', function(e) {
                e.preventDefault();
                self.clearLogs();
            });
            
            // View transaction details
            $(document).on('click', '.view-details', function(e) {
                e.preventDefault();
                var transactionData = $(this).data('transaction');
                self.showTransactionDetails(transactionData);
            });
            
            // Settings form validation
            $('form').on('submit', function() {
                return self.validateSettings($(this));
            });
            
            // Environment change handler
            $(document).on('change', '#woocommerce_nass_gateway_environment', function() {
                self.handleEnvironmentChange($(this).val());
            });
            
            // Terminal ID validation
            $(document).on('input', '#woocommerce_nass_gateway_terminal_id', function() {
                self.validateTerminalId($(this));
            });
            
            // URL validation
            $(document).on('input', '#woocommerce_nass_gateway_gateway_url', function() {
                self.validateUrl($(this));
            });
            
            // Auto-save settings
            $(document).on('change', '.nass-gateway-setting', function() {
                self.autoSaveSettings();
            });
            
            // Refresh logs button
            $(document).on('click', '#refresh-logs', function(e) {
                e.preventDefault();
                self.refreshLogs();
            });
        },
        
        /**
         * Test connection to NASS gateway
         */
        testConnection: function() {
            var $button = $('#test-connection');
            var originalText = $button.text();
            var $icon = $button.find('.dashicons');
            
            // Update button state
            $button.prop('disabled', true)
                   .text('Testing...')
                   .prepend('<span class="spinner is-active" style="float: none; margin: 0 5px 0 0;"></span>');
            
            // Show loading state
            this.showLoadingState($button.closest('.nass-actions'));
            
            $.ajax({
                url: nass_admin_params.ajax_url,
                type: 'POST',
                data: {
                    action: 'nass_test_connection',
                    nonce: nass_admin_params.nonce
                },
                timeout: 30000,
                success: function(response) {
                    if (response.success) {
                        NassAdmin.showNotice('success', response.data.message);
                        NassAdmin.updateConnectionStatus(true);
                    } else {
                        NassAdmin.showNotice('error', response.data.message);
                        NassAdmin.updateConnectionStatus(false);
                    }
                },
                error: function(xhr, status, error) {
                    var message = 'Connection test failed. ';
                    if (status === 'timeout') {
                        message += 'Request timed out.';
                    } else {
                        message += 'Please check your network connection and try again.';
                    }
                    NassAdmin.showNotice('error', message);
                    NassAdmin.updateConnectionStatus(false);
                },
                complete: function() {
                    // Restore button state
                    $button.prop('disabled', false)
                           .text(originalText)
                           .find('.spinner').remove();
                    
                    NassAdmin.hideLoadingState($button.closest('.nass-actions'));
                }
            });
        },
        
        /**
         * Clear debug logs
         */
        clearLogs: function() {
            if (!confirm('Are you sure you want to clear all logs? This action cannot be undone.')) {
                return;
            }
            
            var $button = $('#clear-logs');
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Clearing...');
            
            $.ajax({
                url: nass_admin_params.ajax_url,
                type: 'POST',
                data: {
                    action: 'nass_clear_logs',
                    nonce: nass_admin_params.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $('.nass-log-viewer textarea').val('');
                        NassAdmin.showNotice('success', response.data.message);
                        NassAdmin.updateLogStats(0);
                    } else {
                        NassAdmin.showNotice('error', 'Failed to clear logs.');
                    }
                },
                error: function() {
                    NassAdmin.showNotice('error', 'Failed to clear logs. Please try again.');
                },
                complete: function() {
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },
        
        /**
         * Show transaction details in modal
         */
        showTransactionDetails: function(transactionData) {
            try {
                var data = typeof transactionData === 'string' ? JSON.parse(transactionData) : transactionData;
                var html = '<table class="wp-list-table widefat fixed striped">';
                html += '<thead><tr><th>Field</th><th>Value</th></tr></thead><tbody>';
                
                for (var key in data) {
                    if (data.hasOwnProperty(key)) {
                        html += '<tr><td><strong>' + this.formatKey(key) + '</strong></td><td>' + 
                               this.formatValue(data[key]) + '</td></tr>';
                    }
                }
                
                html += '</tbody></table>';
                
                $('#transaction-details-content').html(html);
                $('#transaction-details-modal').show();
                
            } catch (e) {
                this.showNotice('error', 'Failed to parse transaction details.');
                console.error('Transaction details error:', e);
            }
        },
        
        /**
         * Initialize modal functionality
         */
        initModal: function() {
            var self = this;
            
            // Close modal on close button click
            $(document).on('click', '.nass-modal-close', function() {
                $('.nass-modal').hide();
            });
            
            // Close modal on outside click
            $(document).on('click', '.nass-modal', function(e) {
                if (e.target === this) {
                    $(this).hide();
                }
            });
            
            // Close modal on escape key
            $(document).on('keydown', function(e) {
                if (e.keyCode === 27) {
                    $('.nass-modal').hide();
                }
            });
        },
        
        /**
         * Initialize dashboard functionality
         */
        initDashboard: function() {
            this.initCharts();
            this.initRealTimeUpdates();
            this.checkSystemStatus();
        },
        
        /**
         * Initialize charts (if chart library is available)
         */
        initCharts: function() {
            // Placeholder for chart initialization
            // You could integrate Chart.js or similar library here
            $('.nass-stats-grid .nass-stat-card').each(function() {
                $(this).hover(
                    function() { $(this).addClass('hover'); },
                    function() { $(this).removeClass('hover'); }
                );
            });
        },
        
        /**
         * Initialize real-time updates
         */
        initRealTimeUpdates: function() {
            // Auto-refresh transaction data every 30 seconds
            if (window.location.href.indexOf('tab=transactions') > -1) {
                setInterval(function() {
                    if (!document.hidden) {
                        NassAdmin.refreshTransactionData();
                    }
                }, 30000);
            }
            
            // Auto-refresh logs every 15 seconds
            if (window.location.href.indexOf('tab=logs') > -1) {
                setInterval(function() {
                    if (!document.hidden) {
                        NassAdmin.refreshLogs();
                    }
                }, 15000);
            }
        },
        
        /**
         * Validate settings form
         */
        validateSettings: function($form) {
            var isValid = true;
            var errors = [];
            
            // Only validate if it's the NASS gateway settings form
            if ($form.find('#woocommerce_nass_gateway_terminal_id').length === 0) {
                return true;
            }
            
            // Check required fields
            var requiredFields = {
                'terminal_id': 'Terminal ID',
                'username': 'Username',
                'password': 'Password',
                'gateway_url': 'Gateway URL'
            };
            
            for (var field in requiredFields) {
                var $field = $('#woocommerce_nass_gateway_' + field);
                if ($field.length && !$field.val().trim()) {
                    isValid = false;
                    errors.push('Please fill in the ' + requiredFields[field] + ' field.');
                    $field.addClass('error');
                } else {
                    $field.removeClass('error');
                }
            }
            
            // Validate terminal ID format
            var terminalId = $('#woocommerce_nass_gateway_terminal_id').val();
            if (terminalId && !/^\d{8}$/.test(terminalId)) {
                isValid = false;
                errors.push('Terminal ID must be exactly 8 digits.');
                $('#woocommerce_nass_gateway_terminal_id').addClass('error');
            }
            
            // Validate gateway URL
            var gatewayUrl = $('#woocommerce_nass_gateway_gateway_url').val();
            if (gatewayUrl && !this.isValidUrl(gatewayUrl)) {
                isValid = false;
                errors.push('Please enter a valid gateway URL.');
                $('#woocommerce_nass_gateway_gateway_url').addClass('error');
            }
            
            if (!isValid) {
                this.showNotice('error', errors.join('<br>'));
                return false;
            }
            
            return true;
        },
        
        /**
         * Handle environment change
         */
        handleEnvironmentChange: function(environment) {
            var $notice = $('.environment-notice');
            $notice.remove();
            
            if (environment === 'production') {
                var notice = '<div class="notice notice-warning environment-notice">' +
                           '<p><strong>Production Mode:</strong> Make sure you have the correct production credentials and SSL is enabled.</p>' +
                           '</div>';
                $('.nass-gateway-settings').prepend(notice);
            } else {
                var notice = '<div class="notice notice-info environment-notice">' +
                           '<p><strong>Sandbox Mode:</strong> Use test credentials and test card numbers for testing.</p>' +
                           '</div>';
                $('.nass-gateway-settings').prepend(notice);
            }
        },
        
        /**
         * Validate terminal ID
         */
        validateTerminalId: function($field) {
            var value = $field.val();
            var $feedback = $field.next('.field-feedback');
            
            if ($feedback.length === 0) {
                $feedback = $('<span class="field-feedback"></span>');
                $field.after($feedback);
            }
            
            if (value === '') {
                $feedback.html('').removeClass('error success');
            } else if (!/^\d{8}$/.test(value)) {
                $feedback.html('Must be exactly 8 digits').addClass('error').removeClass('success');
            } else {
                $feedback.html('Valid format').addClass('success').removeClass('error');
            }
        },
        
        /**
         * Validate URL
         */
        validateUrl: function($field) {
            var value = $field.val();
            var $feedback = $field.next('.field-feedback');
            
            if ($feedback.length === 0) {
                $feedback = $('<span class="field-feedback"></span>');
                $field.after($feedback);
            }
            
            if (value === '') {
                $feedback.html('').removeClass('error success');
            } else if (!this.isValidUrl(value)) {
                $feedback.html('Invalid URL format').addClass('error').removeClass('success');
            } else {
                $feedback.html('Valid URL').addClass('success').removeClass('error');
            }
        },
        
        /**
         * Auto-save settings
         */
        autoSaveSettings: function() {
            // Implement auto-save functionality if needed
            $('.auto-save-indicator').show().fadeOut(2000);
        },
        
        /**
         * Refresh logs
         */
        refreshLogs: function() {
            var $textarea = $('.nass-log-viewer textarea');
            var $button = $('#refresh-logs');
            
            if ($button.length) {
                $button.prop('disabled', true).text('Refreshing...');
            }
            
            // Reload the current page to refresh logs
            setTimeout(function() {
                window.location.reload();
            }, 500);
        },
        
        /**
         * Refresh transaction data
         */
        refreshTransactionData: function() {
            // Implement AJAX refresh of transaction table
            $.ajax({
                url: nass_admin_params.ajax_url,
                type: 'POST',
                data: {
                    action: 'nass_refresh_transactions',
                    nonce: nass_admin_params.nonce
                },
                success: function(response) {
                    if (response.success && response.data.html) {
                        $('.nass-transactions table tbody').html(response.data.html);
                    }
                },
                error: function() {
                    console.log('Failed to refresh transaction data');
                }
            });
        },
        
        /**
         * Check system status
         */
        checkSystemStatus: function() {
            // Check system requirements and display warnings if needed
            this.checkSSLStatus();
            this.checkWooCommerceVersion();
            this.checkPHPVersion();
        },
        
        /**
         * Check SSL status
         */
        checkSSLStatus: function() {
            if (location.protocol !== 'https:' && $('#woocommerce_nass_gateway_environment').val() === 'production') {
                this.showNotice('warning', 'SSL certificate is required for production mode.');
            }
        },
        
        /**
         * Check WooCommerce version
         */
        checkWooCommerceVersion: function() {
            // This would typically be done server-side
        },
        
        /**
         * Check PHP version
         */
        checkPHPVersion: function() {
            // This would typically be done server-side
        },
        
        /**
         * Update connection status indicator
         */
        updateConnectionStatus: function(connected) {
            var $indicator = $('.connection-status');
            if ($indicator.length === 0) {
                $indicator = $('<span class="connection-status"></span>');
                $('#test-connection').after($indicator);
            }
            
            if (connected) {
                $indicator.html('<span class="dashicons dashicons-yes-alt"></span> Connected')
                          .removeClass('disconnected').addClass('connected');
            } else {
                $indicator.html('<span class="dashicons dashicons-dismiss"></span> Disconnected')
                          .removeClass('connected').addClass('disconnected');
            }
        },
        
        /**
         * Update log statistics
         */
        updateLogStats: function(count) {
            $('.log-count').text(count + ' entries');
        },
        
        /**
         * Show loading state
         */
        showLoadingState: function($container) {
            $container.addClass('nass-loading');
        },
        
        /**
         * Hide loading state
         */
        hideLoadingState: function($container) {
            $container.removeClass('nass-loading');
        },
        
        /**
         * Show admin notice
         */
        showNotice: function(type, message) {
            var noticeClass = type === 'success' ? 'notice-success' : 
                             type === 'warning' ? 'notice-warning' : 'notice-error';
            var $notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>');
            
            // Remove existing notices
            $('.notice.nass-notice').remove();
            $notice.addClass('nass-notice');
            
            $('.wrap h1').after($notice);
            
            // Auto dismiss after 5 seconds
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $notice.remove();
                });
            }, 5000);
            
            // Scroll to top
            $('html, body').animate({
                scrollTop: 0
            }, 300);
        },
        
        /**
         * Format object key for display
         */
        formatKey: function(key) {
            return key.replace(/([A-Z])/g, ' $1')
                     .replace(/^./, function(str) { return str.toUpperCase(); })
                     .replace(/_/g, ' ');
        },
        
        /**
         * Format value for display
         */
        formatValue: function(value) {
            if (typeof value === 'object' && value !== null) {
                return '<pre>' + JSON.stringify(value, null, 2) + '</pre>';
            }
            if (typeof value === 'boolean') {
                return value ? 'Yes' : 'No';
            }
            return String(value);
        },
        
        /**
         * Validate URL format
         */
        isValidUrl: function(string) {
            try {
                new URL(string);
                return true;
            } catch (_) {
                return false;
            }
        }
    };
    
    // Initialize admin functionality
    NassAdmin.init();
    
    // Handle page visibility change
    $(document).on('visibilitychange', function() {
        if (!document.hidden) {
            // Page became visible, refresh data if needed
            if (window.location.href.indexOf('nass-payment-gateway') > -1) {
                NassAdmin.checkSystemStatus();
            }
        }
    });
});