<?php
/**
 * NASS Payment Gateway Test Checkout
 * Place this file in your WordPress root directory
 * Access it via: https://yoursite.com/checkout-test-rrn.php
 */

// Load WordPress to use its functions
require_once('wp-load.php');

// Configuration
$config = array(
    // Old credentials - commented out
    // 'username' => 'info@epicstore-key.com',
    // 'password' => 'Epicstore@2025?',
    // 'terminal_id' => '00099619',
    
    // New production credentials
    'username' => 'EPIC_STORE852',
    'password' => 'EPIC_STORE852',
    'terminal_id' => '00029442',
    
    'api_url' => 'https://uat-gateway.nass.iq:9746/',
    'gateway_url' => 'https://3dsecure.nass.iq:9746/gateway/'
);

// Handle return from NASS
if (isset($_GET['status'])) {
    echo '<div style="max-width: 800px; margin: 50px auto; font-family: Arial, sans-serif;">';
    echo '<h1>Payment Return from NASS</h1>';
    echo '<div style="background: #f0f0f0; padding: 20px; border-radius: 5px; margin: 20px 0;">';
    echo '<h2>Return Parameters:</h2>';
    echo '<pre>' . print_r($_GET, true) . '</pre>';
    echo '</div>';
    
    if ($_GET['status'] === 'success') {
        echo '<div style="background: #d4edda; color: #155724; padding: 15px; border-radius: 5px;">';
        echo '<h3>✅ Payment Successful!</h3>';
        echo '<p>RRN (Payment ID): <strong>' . ($_GET['rrn'] ?? 'Not provided') . '</strong></p>';
        echo '<p>Order ID: <strong>' . ($_GET['orderId'] ?? 'Not provided') . '</strong></p>';
        echo '<p>Amount: <strong>' . ($_GET['amount'] ?? 'Not provided') . '</strong></p>';
        echo '</div>';
    } else {
        echo '<div style="background: #f8d7da; color: #721c24; padding: 15px; border-radius: 5px;">';
        echo '<h3>❌ Payment Failed</h3>';
        echo '<p>Status: <strong>' . $_GET['status'] . '</strong></p>';
        echo '<p>Please check the parameters above for error details.</p>';
        echo '</div>';
    }
    
    echo '<div style="margin-top: 20px;">';
    echo '<a href="checkout-test-rrn.php" style="background: #007cba; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;">Try Another Payment</a>';
    echo '</div>';
    echo '</div>';
    exit;
}

// Function to authenticate with NASS
function authenticate_nass($config) {
    $auth_url = $config['api_url'] . 'auth/merchant/login';
    
    $response = wp_remote_post($auth_url, array(
        'body' => json_encode(array(
            'username' => $config['username'],
            'password' => $config['password']
        )),
        'headers' => array(
            'Content-Type' => 'application/json'
        ),
        'sslverify' => false
    ));
    
    if (is_wp_error($response)) {
        return array('error' => $response->get_error_message());
    }
    
    $body = json_decode(wp_remote_retrieve_body($response), true);
    
    if ($body['success'] && isset($body['data']['access_token'])) {
        return array('token' => $body['data']['access_token']);
    }
    
    return array('error' => 'Authentication failed');
}

// Function to create transaction
function create_transaction($config, $token, $order_data) {
    $transaction_url = $config['api_url'] . 'transaction';
    
    $response = wp_remote_post($transaction_url, array(
        'body' => json_encode($order_data),
        'headers' => array(
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $token
        ),
        'sslverify' => false
    ));
    
    if (is_wp_error($response)) {
        return array('error' => $response->get_error_message());
    }
    
    $body = json_decode(wp_remote_retrieve_body($response), true);
    
    if ($body['success'] && isset($body['data']['url'])) {
        return array('payment_url' => $body['data']['url'], 'response' => $body);
    }
    
    return array('error' => 'Transaction creation failed', 'response' => $body);
}

// Function to check transaction status
function check_transaction_status($config, $token, $order_id) {
    $status_url = $config['api_url'] . 'transaction/' . $order_id . '/checkStatus';
    
    $response = wp_remote_get($status_url, array(
        'headers' => array(
            'Authorization' => 'Bearer ' . $token
        ),
        'sslverify' => false
    ));
    
    if (is_wp_error($response)) {
        return array('error' => $response->get_error_message());
    }
    
    $body = json_decode(wp_remote_retrieve_body($response), true);
    return $body;
}

// Process form submission
$message = '';
$debug_info = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check if this is a status check
    if (isset($_POST['check_status']) && isset($_POST['order_id'])) {
        $auth_result = authenticate_nass($config);
        if (!isset($auth_result['error'])) {
            $status_result = check_transaction_status($config, $auth_result['token'], $_POST['order_id']);
            $message = '<div style="background: #e3f2fd; color: #0c5460; padding: 15px; border-radius: 5px;">';
            $message .= '<h3>Transaction Status for Order: ' . $_POST['order_id'] . '</h3>';
            $message .= '<pre>' . print_r($status_result, true) . '</pre>';
            $message .= '</div>';
        }
    } else {
        // Generate test order data
        $order_id = strval(random_int(1000000000, 9999999999)); // 10-digit unique random number
        $amount = $_POST['amount'] ?? '1000.00';
        $description = $_POST['description'] ?? 'Test Payment';
        
        // Log the attempt
        error_log('NASS Test Checkout - Creating order: ' . $order_id . ' for amount: ' . $amount);
        
        // Step 1: Authenticate
        $auth_result = authenticate_nass($config);
        
        if (isset($auth_result['error'])) {
            $message = '<div style="background: #f8d7da; color: #721c24; padding: 15px; border-radius: 5px;">Authentication Error: ' . $auth_result['error'] . '</div>';
        } else {
            // Step 2: Create transaction
            $return_url = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . '/checkout-test-rrn.php';
            
            $transaction_data = array(
                'orderId' => $order_id,
                'orderDesc' => $description,
                'amount' => number_format($amount, 2, '.', ''),
                'currency' => '368', // IQD
                'transactionType' => '1',
                'backRef' => $return_url,
                'notifyUrl' => $return_url
            );
            
            $transaction_result = create_transaction($config, $auth_result['token'], $transaction_data);
            
            if (isset($transaction_result['error'])) {
                $message = '<div style="background: #f8d7da; color: #721c24; padding: 15px; border-radius: 5px;">Transaction Error: ' . $transaction_result['error'] . '</div>';
                $debug_info = '<pre>' . print_r($transaction_result['response'] ?? '', true) . '</pre>';
            } else {
                // Store order ID in session for status check
                session_start();
                $_SESSION['last_order_id'] = $order_id;
                
                // Show debug info before redirect
                error_log('NASS Test Checkout - Redirecting to: ' . $transaction_result['payment_url']);
                error_log('NASS Test Checkout - Full response: ' . print_r($transaction_result['response'], true));
                
                // Redirect to payment URL
                header('Location: ' . $transaction_result['payment_url']);
                exit;
            }
        }
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>NASS Payment Test Checkout</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            background: #f5f5f5;
            margin: 0;
            padding: 20px;
        }
        .container {
            max-width: 600px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            text-align: center;
            margin-bottom: 30px;
        }
        .form-group {
            margin-bottom: 20px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: #555;
        }
        input[type="text"], input[type="number"] {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
            box-sizing: border-box;
        }
        button {
            width: 100%;
            padding: 15px;
            background: #007cba;
            color: white;
            border: none;
            border-radius: 5px;
            font-size: 18px;
            cursor: pointer;
            transition: background 0.3s;
        }
        button:hover {
            background: #005a87;
        }
        .test-info {
            background: #e3f2fd;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .test-cards {
            background: #fff3cd;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .test-cards h3 {
            margin-top: 0;
            color: #856404;
        }
        .test-cards code {
            background: #ffeaa7;
            padding: 2px 5px;
            border-radius: 3px;
            font-family: monospace;
        }
        .config-info {
            background: #f0f0f0;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        .debug {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin-top: 20px;
            font-size: 12px;
            overflow-x: auto;
        }
        .credential-info {
            background: #d1ecf1;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border: 1px solid #bee5eb;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🏦 NASS Payment Test Checkout</h1>
        
        <div class="config-info">
            <strong>Current Configuration:</strong><br>
            API URL: <?php echo $config['api_url']; ?><br>
            Terminal ID: <?php echo $config['terminal_id']; ?><br>
            Environment: UAT (Testing)
        </div>
        
        <div class="credential-info">
            <strong>🔑 New Production Credentials:</strong><br>
            Username: <code><?php echo $config['username']; ?></code><br>
            Password: <code><?php echo $config['password']; ?></code><br>
            Terminal ID: <code><?php echo $config['terminal_id']; ?></code><br>
            <small style="color: #666;">Old credentials (info@epicstore-key.com) have been replaced</small>
        </div>
        
        <div class="test-info">
            <strong>ℹ️ How to test:</strong><br>
            1. Enter an amount and description below<br>
            2. Click "Proceed to Payment"<br>
            3. You'll be redirected to NASS payment page<br>
            4. Use test cards to complete payment<br>
            5. You'll return here with the payment result
        </div>
        
        <div class="test-cards">
            <h3>🔐 UAT Test Cards:</h3>
            <strong>Visa:</strong> <code>4761349999000039</code> | Expiry: <code>12/31</code><br>
            <strong>Mastercard:</strong> <code>5185520050000010</code> | Expiry: <code>12/25</code> | CVV: <code>356</code>
        </div>
        
        <?php if ($message): ?>
            <?php echo $message; ?>
            <?php if ($debug_info): ?>
                <div class="debug">
                    <strong>Debug Information:</strong>
                    <?php echo $debug_info; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <form method="POST">
            <div class="form-group">
                <label for="amount">Amount (IQD):</label>
                <input type="number" id="amount" name="amount" value="1000.00" step="0.01" required>
            </div>
            
            <div class="form-group">
                <label for="description">Description:</label>
                <input type="text" id="description" name="description" value="Test Payment <?php echo date('Y-m-d H:i:s'); ?>" required>
            </div>
            
            <button type="submit">💳 Proceed to Payment</button>
        </form>
        
        <?php 
        session_start();
        if (isset($_SESSION['last_order_id'])): 
        ?>
        <div style="margin-top: 20px; padding: 15px; background: #f0f0f0; border-radius: 5px;">
            <form method="POST" style="margin: 0;">
                <input type="hidden" name="check_status" value="1">
                <input type="hidden" name="order_id" value="<?php echo $_SESSION['last_order_id']; ?>">
                <button type="submit" style="background: #28a745;">🔍 Check Status of Last Order (<?php echo $_SESSION['last_order_id']; ?>)</button>
            </form>
        </div>
        <?php endif; ?>
        
        <div style="margin-top: 30px; padding: 20px; background: #f8f9fa; border-radius: 5px;">
            <h3>🐛 Debugging Tips:</h3>
            <ol style="margin: 10px 0; padding-left: 20px;">
                <li><strong>Empty RRN:</strong> This means the payment didn't complete successfully at NASS</li>
                <li><strong>Common reasons for failure:</strong>
                    <ul>
                        <li>Invalid card number or expiry date</li>
                        <li>Terminal ID (<?php echo $config['terminal_id']; ?>) not active in UAT</li>
                        <li>Amount format issues</li>
                        <li>3D Secure authentication failed</li>
                    </ul>
                </li>
                <li><strong>Try different amounts:</strong> 1000, 5000, 10000</li>
                <li><strong>Check WordPress debug log</strong> at: <code>wp-content/debug.log</code></li>
            </ol>
        </div>
        
        <div style="margin-top: 30px; text-align: center; color: #666; font-size: 14px;">
            <p>This is a test page for NASS Payment Gateway integration</p>
            <p><a href="<?php echo home_url(); ?>">← Back to website</a></p>
        </div>
    </div>
</body>
</html>